/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
import { ErrorInformation, ErrorReporter } from "./advanced";
/**
 * The class you must extend to make a custom reporting channel.
 *
 * @export
 * @abstract
 * @class ReportingChannel
 */
export declare abstract class ReportingChannel {
    reportOnFailure: boolean;
    timeout: number;
    /**
     * Reports the error.
     *
     * @abstract
     * @param {ErrorInformation} error - Information on the error to be reported
     * @returns {Promise<any>} - resolves when it has successfully been reported or rejects when it fails.
     * Can be synchronous or asynchronous.
     */
    abstract report(error: ErrorInformation): Promise<any>;
    /**
     * Creates an instance of ReportingChannel.
     *
     * @param {boolean} [reportOnFailure]
     * @param {number} [timeoutMS]
     */
    constructor(reportOnFailure?: boolean, timeoutMS?: number);
    registerListeners(reporter: ErrorReporter): void;
}
/**
 * An Error reporting channel that simply prints the raw stack trace to the console.
 */
export declare class ConsoleReporter extends ReportingChannel {
    constructor();
    report(error: ErrorInformation): Promise<void>;
}
/**
 * An error reporting channel which writes the stack trace to a log file.
 * Depending on the options given to the constructor, you can set the fileName,
 * file Location, and whether the stack should be stripped of full paths and PII before it is written.
 */
export declare class FileReporter extends ReportingChannel {
    /**
     * creates a reporter which will create or append to the specified file.
     * The contents is the stack trace of a crash.
     * path: recommend absolute, or else it will be relative to the cwd,
     * which may change depending on how it is launched.
     * filename: the name of the file.
     * strip stack: whether to strip the stack trace of absolute paths, suitable for upload, or leave it alone.
     */
    constructor(filePath: string, fileName: string, stripStack: boolean);
    report(error: ErrorInformation): Promise<string>;
}
